<?php

class ImportAction extends CAction
{
	const STATE_CSV = 'admin.zlecenia.import.spreadsheet';
	const STATE_ZIP = 'admin.zlecenia.import.archive';
	const STATE_OFFSET = 'admin.zlecenia.import.offset';
	const STATE_GIFT = 'admin.zlecenia.import.gift';
	const LENGTH = 10;

	private static $ENCODEINGS = array(
		'UTF-8' => 'UTF-8',
		'CP1250' => 'Windows Środkowoeuropejski (CP1250)',
		'ISO-8859-2' => 'ISO Środkowoeuropejski (ISO-8859-2)'
	);

	private function uploadFile(CUploadedFile $file)
	{
		$fileType = '';
		try
		{
			if ($file->hasError)
				throw new Exception($file->error);

			$mime = CFileHelper::getMimeType($file->getTempName());
			if (empty($mime))
				$mime = $file->getType();
			if (empty($mime))
				$mime = CFileHelper::getMimeTypeByExtension($file->getName());

			$encoding = '';

			if ($mime == 'application/zip' || $mime == 'application/x-zip-compressed')
			{
				$state = self::STATE_ZIP;
				$fileType = 'zip';
			}
			else
			{
				Yii::import('ext.yiiexcel.YiiExcel', true);
				YiiExcel::init();
				$fileType = PHPExcel_IOFactory::identify($file->getTempName());
				if (!empty($fileType) && $fileType != 'Excel2007')
				{
					$state = self::STATE_CSV;
					$encoding = strtoupper(filter_input(INPUT_GET, 'csv-encodeing', FILTER_SANITIZE_STRING));
					if (!isset(self::$ENCODEINGS[$encoding]))
						$encoding = 'CP1250';
				}
				else
					throw new Exception('Format "' . $mime . '" nie jest wspierany.');
			}

			$currState = Yii::app()->user->getState($state);
			if (!empty($currState))
			{
				if (file_exists($state['file']))
					unlink($state['file']);
			}

			$filePath = $file->getTempName() . '-' . md5(uniqid());
			if (!(@$file->saveAs($filePath)))
				throw new Exception('Nie można zapisać pliku.');

			$newState = array(
				'name' => $file->getName(),
				'size' => $file->getSize(),
				'mime' => $mime,
				'type' => $fileType,
				'file' => $filePath,
				'encoding' => $encoding
			);
			Yii::app()->user->setState($state, $newState);
		}
		catch (Exception $e)
		{
			$type = $fileType == 'zip' ? $fileType : 'csv';
			return array($type => $e->getMessage());
		}
	}

	private function cancel()
	{
		$csv = Yii::app()->user->getState(self::STATE_CSV);
		if (is_array($csv) && isset($csv['file']) && file_exists($csv['file']))
			unlink($csv['file']);
		$zip = Yii::app()->user->getState(self::STATE_ZIP);
		if (is_array($zip) && isset($zip['file']) && file_exists($zip['file']))
			unlink($zip['file']);
		$gift = Yii::app()->user->getState(self::STATE_GIFT);
		if (is_array($gift) && isset($gift['path']) && file_exists($gift['path']))
			unlink($gift['path']);
		Yii::app()->user->setState(self::STATE_CSV, null);
		Yii::app()->user->setState(self::STATE_OFFSET, null);
		Yii::app()->user->setState(self::STATE_ZIP, null);
		Yii::app()->user->setState(self::STATE_GIFT, null);
	}

	private function processQueue()
	{
		header('Content-type: text/json');
		$result = array();

		require_once 'ImageReader.php';
		$images = new ImageReader;
		try
		{
			$csv = Yii::app()->user->getState(self::STATE_CSV);
			if (empty($csv))
				throw new Exception('Nie wysłano żadnego pliku do ptrzetworzenia.');

			$offset = Yii::app()->user->getState(self::STATE_OFFSET);
			if (empty($offset))
				$offset = 2;

			require_once 'ReadFilter.php';

			$objReader = PHPExcel_IOFactory::createReader($csv['type']);
			if (method_exists($objReader, 'listWorksheetNames'))
			{
				$workSheetNames = $objReader->listWorksheetNames($csv['file']);
				$objReader->setLoadSheetsOnly($workSheetNames[0]);
			}
			$info = $objReader->listWorksheetInfo($csv['file']);
			$total = $info[0]['totalRows'];
			$readFilter = new ReadFilter($offset, self::LENGTH);
			$objReader->setReadFilter($readFilter);
			if ($csv['type'] === 'CSV')
			{
				$objReader->setInputEncoding($csv['encoding']);
				$objReader->setDelimiter(';');
			}

			$objPHPExcel = $objReader->load($csv['file']);

			$sheet = $objPHPExcel->getSheet(0);

			$images->fetchFromSpreadSheet($sheet, $readFilter);
			$images->fetchFromZip(self::STATE_ZIP);

			$processed = 0;
			$HEADER = array();

			$gift = new GiftWriter(self::STATE_GIFT);

			//process rows
			foreach ($sheet->getRowIterator() as $row)
			{
				$rowNum = $row->getRowIndex();
				if ($rowNum == 1)
				{
					$cellIterator = $row->getCellIterator();
					$cellIterator->setIterateOnlyExistingCells(true);
					foreach ($cellIterator as $cell)
					{
						$value = $cell->getValue();
						$column = $cell->getColumn();
						$HEADER[$column] = strtolower(trim($value));
					}
				}
				else if ($readFilter->rowAllowed($rowNum))
				{
					$zlecenie = new Zlecenie('import');
					$odpowiedzi = array();
					$lp = 0;
					$platforma = false;

					if ($processed < $rowNum)
						$processed = $rowNum;
					$cellIterator = $row->getCellIterator();
					$cellIterator->setIterateOnlyExistingCells(true);
					foreach ($cellIterator as $cell)
					{
						$column = $cell->getColumn();
						if (isset($HEADER[$column]))
						{
							$fieldName = $HEADER[$column];
							$value = $cell->getValue();

							static $fieldMapping = array(
								'dziedzina' => 'idDziedziny',
								'poziom' => 'poziom',
								'kategoria' => 'kategoria',
								'czas' => 'czas',
								'cena' => 'cena',
								'kara' => 'kara',
								'czyprzetarg' => 'czyPrzetarg',
								'tresc' => 'tresc',
								'jednostka' => 'jednostka',
								'czyegzamin' => 'czyEgzamin',
								'jezyk' => 'jezyk',
							);

							if ($zlecenie->hasProperty($fieldName))
								$zlecenie->$fieldName = $value;
							else if (isset($fieldMapping[$fieldName])) {
								$theField = $fieldMapping[$fieldName];
								$zlecenie->$theField = $value;
							}
							else if ($fieldName == 'lp')
								$lp = $value;
							else if ($fieldName == 'czyplatforma')
								$platforma = $value;
							else if (strpos($fieldName, 'odpowiedz') !== false) {
								$value = trim($value);
								if (!empty($value))
									$odpowiedzi[] = $value;
							}
							else {
								$result['entres'][$rowNum]['error'] = 'Nieznana kolumna "' . $HEADER[$column] . '".';
								continue;
							}
						}
					}

					//zapisz zlecenie (i obrazek)
					$zlecenie->urlObrazka = $images->storeImage($lp, $rowNum);

					$result['entres'][$rowNum] = array('lp'=>$lp);

					try
					{
						if (!$zlecenie->save())
						{
							$error = '';
							foreach ($zlecenie->errors as $errors)
								$error .= (empty($error)?'':' ') . implode(' ', $errors);
							if (empty($error))
								throw new Exception('Nie można dodać zlecenia do bazy.');
							else
								throw new Exception($error);
						}
					}
					catch (Exception $ex)
					{
						if (file_exists($zlecenie->urlObrazka))
							unlink($zlecenie->urlObrazka);
						$result['entres'][$rowNum]['error'] = $ex->getMessage();
						continue;
					}

					//zapisz odpowiedzi
					foreach ($odpowiedzi as $odpowiedz)
					{
						$ODP = new ZlecenieOdpowiedz;
						$ODP->idZlecenia = $zlecenie->id;
						$ODP->odpowiedz = $odpowiedz;
						$ODP->save();
					}

					if ($platforma)
						$gift->addQuestion($zlecenie, $odpowiedzi);

					$result['entres'][$rowNum]['ok'] = 1;
				}
			}

			$result['done'] = $processed >= $total ? 1 : 0;
			$result['total'] = $total;
			$result['processed'] = $processed;

			$objPHPExcel->disconnectWorksheets();
			unset($objPHPExcel);
			$images->close();
			$gift->close();

			if ($result['done'])
			{
				$gift->send();
				$this->cancel();
			}
			else
			{
				$offset += self::LENGTH;
				Yii::app()->user->setState(self::STATE_OFFSET, $offset);
			}
		}
		catch (Exception $e)
		{
			$images->close();
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function run($cancel = null)
	{
		if ($cancel)
		{
			$this->cancel();
		}

		$errors = array();

		$upload = filter_input(INPUT_POST, 'upload', FILTER_SANITIZE_STRING);
		if (!empty($upload))
		{
			$files = CUploadedFile::getInstancesByName('files');
			foreach ($files as $file)
				$errors = $this->uploadFile($file);
			if (count($files) == 0)
				$errors[] = 'Nie wysłano żdanych plików na serwer.';
		}

		$process = filter_input(INPUT_POST, 'process', FILTER_VALIDATE_INT);
		if ($process)
		{
			$this->processQueue();
			return;
		}

		$this->getController()->render('import', array(
			'dataCsv' => Yii::app()->user->getState(self::STATE_CSV),
			'dataZip' => Yii::app()->user->getState(self::STATE_ZIP),
			'encodeings' => self::$ENCODEINGS,
			'errors' => $errors
		));
	}

}
